<?php
session_start();

$configFile = __DIR__ . '/onedrive_config.json';
$syncDir = __DIR__ . '/synced_files';

// Load config
$config = file_exists($configFile) ? json_decode(file_get_contents($configFile), true) : [];

// Handle actions
$action = $_GET['action'] ?? $_POST['action'] ?? '';
$message = '';

// Save API credentials
if ($action === 'save_credentials') {
    $config['client_id'] = trim($_POST['client_id']);
    $config['client_secret'] = trim($_POST['client_secret']);
    $config['tenant_id'] = trim($_POST['tenant_id']) ?: 'common';
    $config['redirect_uri'] = trim($_POST['redirect_uri']);
    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    $message = 'Credentials saved.';
}

// Start OAuth
if ($action === 'login') {
    $authUrl = "https://login.microsoftonline.com/{$config['tenant_id']}/oauth2/v2.0/authorize?" . http_build_query([
        'client_id' => $config['client_id'],
        'response_type' => 'code',
        'redirect_uri' => $config['redirect_uri'],
        'scope' => 'offline_access Files.Read Files.Read.All',
        'response_mode' => 'query'
    ]);
    header("Location: $authUrl");
    exit;
}

// OAuth callback
if (isset($_GET['code'])) {
    $tokenUrl = "https://login.microsoftonline.com/{$config['tenant_id']}/oauth2/v2.0/token";
    $response = file_get_contents($tokenUrl, false, stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query([
                'client_id' => $config['client_id'],
                'client_secret' => $config['client_secret'],
                'code' => $_GET['code'],
                'redirect_uri' => $config['redirect_uri'],
                'grant_type' => 'authorization_code'
            ])
        ]
    ]));
    $tokens = json_decode($response, true);
    if (isset($tokens['access_token'])) {
        $config['access_token'] = $tokens['access_token'];
        $config['refresh_token'] = $tokens['refresh_token'] ?? '';
        $config['token_expires'] = time() + ($tokens['expires_in'] ?? 3600);
        file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
        $message = 'Connected to OneDrive!';
    } else {
        $message = 'Auth failed: ' . ($tokens['error_description'] ?? 'Unknown error');
    }
}

// Refresh token if needed
function refreshToken(&$config, $configFile) {
    if (empty($config['refresh_token'])) return false;
    $tokenUrl = "https://login.microsoftonline.com/{$config['tenant_id']}/oauth2/v2.0/token";
    $response = @file_get_contents($tokenUrl, false, stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query([
                'client_id' => $config['client_id'],
                'client_secret' => $config['client_secret'],
                'refresh_token' => $config['refresh_token'],
                'grant_type' => 'refresh_token'
            ])
        ]
    ]));
    $tokens = json_decode($response, true);
    if (isset($tokens['access_token'])) {
        $config['access_token'] = $tokens['access_token'];
        $config['refresh_token'] = $tokens['refresh_token'] ?? $config['refresh_token'];
        $config['token_expires'] = time() + ($tokens['expires_in'] ?? 3600);
        file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
        return true;
    }
    return false;
}

// API call helper
function apiCall($endpoint, &$config, $configFile) {
    if (($config['token_expires'] ?? 0) < time()) {
        refreshToken($config, $configFile);
    }
    $ctx = stream_context_create([
        'http' => [
            'header' => "Authorization: Bearer {$config['access_token']}"
        ]
    ]);
    $response = @file_get_contents("https://graph.microsoft.com/v1.0$endpoint", false, $ctx);
    return json_decode($response, true);
}

// Get folders
function getFoldersRecursive(&$config, $configFile, $itemId = 'root', $prefix = '') {
    $endpoint = $itemId === 'root'
        ? '/me/drive/root/children'
        : "/me/drive/items/$itemId/children";

    $data = apiCall($endpoint, $config, $configFile);
    $folders = [];

    if (!empty($data['value'])) {
        foreach ($data['value'] as $item) {
            if (isset($item['folder'])) {
                $path = trim($prefix . '/' . $item['name'], '/');
                $folders[] = [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'path' => $path
                ];
                // recurse
                $folders = array_merge(
                    $folders,
                    getFoldersRecursive($config, $configFile, $item['id'], $path)
                );
            }
        }
    }
    return $folders;
}


// Save selected folders
if ($action === 'save_folders') {
    $config['selected_folders'] = $_POST['folders'] ?? [];
    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    $message = 'Folders saved.';
}

// Sync files
function syncFolder($folderId, $localPath, &$config, $configFile) {
    if (!is_dir($localPath)) mkdir($localPath, 0755, true);
    $items = apiCall("/me/drive/items/$folderId/children", $config, $configFile);
    $count = 0;
    if (isset($items['value'])) {
        foreach ($items['value'] as $item) {
            $itemPath = $localPath . '/' . $item['name'];
            if (isset($item['folder'])) {
                $count += syncFolder($item['id'], $itemPath, $config, $configFile);
            } else if (isset($item['@microsoft.graph.downloadUrl'])) {
                $content = @file_get_contents($item['@microsoft.graph.downloadUrl']);
                if ($content !== false) {
                    file_put_contents($itemPath, $content);
                    $count++;
                }
            }
        }
    }
    return $count;
}

if ($action === 'sync') {
    if (!is_dir($syncDir)) mkdir($syncDir, 0755, true);
    $totalFiles = 0;
    foreach ($config['selected_folders'] ?? [] as $folderId => $folderName) {
        $totalFiles += syncFolder($folderId, $syncDir . '/' . $folderName, $config, $configFile);
    }
    $message = "Synced $totalFiles files.";
}

// Disconnect
if ($action === 'disconnect') {
    unset($config['access_token'], $config['refresh_token'], $config['token_expires']);
    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    $message = 'Disconnected.';
}

$isConnected = !empty($config['access_token']) && ($config['token_expires'] ?? 0) > time();
$folders = $isConnected ? getFoldersRecursive($config, $configFile) : [];
?>
<!DOCTYPE html>
<html>
<head>
    <title>OneDrive Sync</title>
    <style>
        body { font-family: sans-serif; max-width: 800px; margin: 40px auto; padding: 20px; }
        input, button { padding: 8px; margin: 5px 0; }
        input[type="text"] { width: 100%; box-sizing: border-box; }
        .section { background: #f5f5f5; padding: 15px; margin: 15px 0; border-radius: 5px; }
        .msg { background: #dff0d8; padding: 10px; margin: 10px 0; border-radius: 3px; }
        .folder-list { max-height: 300px; overflow-y: auto; }
        label { display: block; padding: 5px 0; }
        button { cursor: pointer; background: #0078d4; color: white; border: none; border-radius: 3px; }
        button:hover { background: #106ebe; }
        .danger { background: #d9534f; }
    </style>
</head>
<body>
    <h1>OneDrive Business Sync</h1>
    
    <?php if ($message): ?>
        <div class="msg"><?= htmlspecialchars($message) ?></div>
    <?php endif; ?>

    <div class="section">
        <h3>API Credentials</h3>
        <form method="post">
            <input type="hidden" name="action" value="save_credentials">
            <label>Client ID:<input type="text" name="client_id" value="<?= htmlspecialchars($config['client_id'] ?? '') ?>"></label>
            <label>Client Secret:<input type="text" name="client_secret" value="<?= htmlspecialchars($config['client_secret'] ?? '') ?>"></label>
            <label>Tenant ID (or 'common'):<input type="text" name="tenant_id" value="<?= htmlspecialchars($config['tenant_id'] ?? 'common') ?>"></label>
            <label>Redirect URI:<input type="text" name="redirect_uri" value="<?= htmlspecialchars($config['redirect_uri'] ?? '') ?>"></label>
            <button type="submit">Save Credentials</button>
        </form>
    </div>

    <div class="section">
        <h3>Connection</h3>
        <?php if ($isConnected): ?>
            <p>✅ Connected to OneDrive</p>
            <a href="?action=disconnect"><button class="danger">Disconnect</button></a>
        <?php else: ?>
            <p>❌ Not connected</p>
            <?php if (!empty($config['client_id'])): ?>
                <a href="?action=login"><button>Connect to OneDrive</button></a>
            <?php else: ?>
                <p>Enter API credentials first.</p>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <?php if ($isConnected): ?>
    <div class="section">
        <h3>Select Folders to Sync</h3>
        <form method="post">
            <input type="hidden" name="action" value="save_folders">
            <div class="folder-list">
                <?php foreach ($folders as $folder): ?>
                    <label>
                        <input type="checkbox" name="folders[<?= htmlspecialchars($folder['id']) ?>]" 
                               value="<?= htmlspecialchars($folder['name']) ?>"
                               <?= isset($config['selected_folders'][$folder['id']]) ? 'checked' : '' ?>>
                        <?= htmlspecialchars($folder['name']) ?>
                    </label>
                <?php endforeach; ?>
                <?php if (empty($folders)): ?>
                    <p>No folders found in root.</p>
                <?php endif; ?>
            </div>
            <button type="submit">Save Selection</button>
        </form>
    </div>

    <div class="section">
        <h3>Sync Now</h3>
        <p>Selected: <?= count($config['selected_folders'] ?? []) ?> folder(s)</p>
        <p>Sync to: <code><?= htmlspecialchars($syncDir) ?></code></p>
        <a href="?action=sync"><button>Start Sync</button></a>
    </div>
    <?php endif; ?>

    <div class="section">
        <h3>Setup Instructions</h3>
        <ol>
            <li>Go to <a href="https://portal.azure.com/#blade/Microsoft_AAD_RegisteredApps/ApplicationsListBlade" target="_blank">Azure App Registrations</a></li>
            <li>Create new registration, set redirect URI to this page URL</li>
            <li>Copy Application (client) ID and Directory (tenant) ID</li>
            <li>Under Certificates & secrets, create a new client secret</li>
            <li>Under API permissions, add Microsoft Graph > Files.Read.All</li>
            <li>Enter credentials above and connect</li>
        </ol>
    </div>
</body>
</html>