<?php
session_start();

$configFile = __DIR__ . '/onedrive_config.json';

if (!file_exists($configFile)) {
    die('Config file not found.');
}

$config = json_decode(file_get_contents($configFile), true);

if (empty($config['access_token']) || ($config['token_expires'] ?? 0) < time()) {
    die('Not connected to OneDrive.');
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

/* ---------------- TOKEN ---------------- */

function refreshToken(&$config, $configFile) {
    if (empty($config['refresh_token'])) return false;

    $res = @file_get_contents(
        "https://login.microsoftonline.com/{$config['tenant_id']}/oauth2/v2.0/token",
        false,
        stream_context_create([
            'http' => [
                'method'  => 'POST',
                'header'  => 'Content-Type: application/x-www-form-urlencoded',
                'content' => http_build_query([
                    'client_id'     => $config['client_id'],
                    'client_secret' => $config['client_secret'],
                    'refresh_token' => $config['refresh_token'],
                    'grant_type'    => 'refresh_token'
                ])
            ]
        ])
    );

    $t = json_decode($res, true);
    if (!empty($t['access_token'])) {
        $config['access_token']  = $t['access_token'];
        $config['refresh_token'] = $t['refresh_token'] ?? $config['refresh_token'];
        $config['token_expires'] = time() + ($t['expires_in'] ?? 3600);
        file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
        return true;
    }
    return false;
}

/* ---------------- API ---------------- */

function apiCall($endpoint, &$config, $configFile) {
    if (($config['token_expires'] ?? 0) < time()) {
        refreshToken($config, $configFile);
    }

    $ctx = stream_context_create([
        'http' => [
            'header' => "Authorization: Bearer {$config['access_token']}"
        ]
    ]);

    $res = @file_get_contents("https://graph.microsoft.com/v1.0$endpoint", false, $ctx);
    return json_decode($res, true);
}

/* ---------------- TREE + SIZE ---------------- */

function fetchTree(&$config, $configFile, $itemId = 'root', $path = '') {
    $endpoint = $itemId === 'root'
        ? '/me/drive/root/children'
        : "/me/drive/items/$itemId/children";

    $data = apiCall($endpoint, $config, $configFile);
    $nodes = [];
    $totalSize = 0;

    foreach ($data['value'] ?? [] as $item) {
        $id   = (string)$item['id'];
        $name = $item['name'];
        $full = trim($path . '/' . $name, '/');

        if (isset($item['folder'])) {
            $child = fetchTree($config, $configFile, $id, $full);
            $totalSize += $child['size'];

            $nodes[] = [
                'id'       => $id,
                'type'     => 'folder',
                'name'     => $name,
                'path'     => $full,
                'size'     => $child['size'],
                'children' => $child['nodes']
            ];
        } else {
            $size = $item['size'] ?? 0;
            $totalSize += $size;

            $nodes[] = [
                'id'   => $id,
                'type' => 'file',
                'name' => $name,
                'path' => $full,
                'size' => $size
            ];
        }
    }

    return ['nodes' => $nodes, 'size' => $totalSize];
}

/* ---------------- REFRESH CACHE ---------------- */

if ($action === 'refresh_folders') {
    $tree = fetchTree($config, $configFile);
    $config['cached_tree'] = $tree['nodes'];
    $config['cached_tree_updated_at'] = time();

    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

/* ---------------- SAVE SELECTION ---------------- */

if ($action === 'save_folders') {
    $existing = $config['selected_folders'] ?? [];
    $incoming = $_POST['folders'] ?? [];

    // Remove unchecked items
    foreach ($existing as $id => $path) {
        if (!isset($incoming[$id])) {
            unset($existing[$id]);
        }
    }

    // Add / update checked items
    foreach ($incoming as $id => $path) {
        $existing[(string)$id] = $path;
    }

    $config['selected_folders'] = $existing;

    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}


/* ---------------- LOAD ---------------- */

$tree     = $config['cached_tree'] ?? [];
$selected = $config['selected_folders'] ?? [];

/* ---------------- UTILS ---------------- */

function fmtSize($bytes) {
    if ($bytes < 1024) return $bytes . ' B';
    if ($bytes < 1048576) return round($bytes / 1024, 1) . ' KB';
    if ($bytes < 1073741824) return round($bytes / 1048576, 1) . ' MB';
    return round($bytes / 1073741824, 2) . ' GB';
}

function renderTree($nodes, $selected, $parent = '') {
    foreach ($nodes as $n) {
$id = $n['id'];                 // RAW — never escape IDs
$safeId = htmlspecialchars($id, ENT_QUOTES);
        $checked = isset($selected[$n['id']]) ? 'checked' : '';
        $size = fmtSize($n['size']);
        $hasChildren = !empty($n['children']);

echo "<div class='node' data-node-id='{$safeId}'>";
        echo "<div class='node-content'>";
        
        if ($hasChildren) {
            echo "<span class='toggle' data-toggle-id='{$id}'>▸</span>";
        } else {
            echo "<span class='toggle-placeholder'></span>";
        }
        
echo "<input
    type='checkbox'
    data-id='{$safeId}'
    data-parent='" . htmlspecialchars($parent, ENT_QUOTES) . "'
    name='folders[{$safeId}]'
    value='" . htmlspecialchars($n['path'], ENT_QUOTES) . "'
    $checked
>";
        echo "<span class='node-label'>";
        echo ($n['type'] === 'folder' ? '📁' : '📄') . " " . htmlspecialchars($n['name']);
        echo " <span class='size'>($size)</span>";
        echo "</span>";
        echo "</div>";

        if ($hasChildren) {
            echo "<div class='children' data-children-of='{$id}' style='display: none;'>";
            renderTree($n['children'], $selected, $n['id']);
            echo "</div>";
        }
        
        echo "</div>";
    }
}
?>
<!DOCTYPE html>
<html>
<head>
<title>Select OneDrive Folders</title>
<style>
body { 
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    max-width: 900px; 
    margin: 40px auto;
    padding: 0 20px;
}

h2 {
    color: #333;
    margin-bottom: 20px;
}

.controls {
    margin-bottom: 20px;
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
    align-items: center;
}

button { 
    padding: 8px 16px;
    background: #0078d4;
    color: #fff;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    transition: background 0.2s;
}

button:hover { 
    background: #106ebe;
}

button:active {
    background: #005a9e;
}

.node {
    margin: 2px 0;
}

.node-content {
    display: flex;
    align-items: center;
    padding: 4px 0;
    cursor: pointer;
    user-select: none;
}

.node-content:hover {
    background: #f5f5f5;
    border-radius: 4px;
}

.toggle {
    cursor: pointer;
    display: inline-block;
    width: 16px;
    height: 16px;
    text-align: center;
    line-height: 16px;
    user-select: none;
    flex-shrink: 0;
    transition: transform 0.15s ease;
}

.toggle.expanded {
    transform: rotate(90deg);
}

.toggle-placeholder {
    display: inline-block;
    width: 16px;
    flex-shrink: 0;
}

.node-content input[type="checkbox"] {
    margin: 0 8px;
    cursor: pointer;
    flex-shrink: 0;
}

.node-label {
    flex: 1;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.size { 
    color: #666;
    font-size: 12px;
    font-weight: normal;
}

.children {
    margin-left: 24px;
}

small {
    color: #666;
    font-size: 12px;
}

.tree-container {
    margin: 20px 0;
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 15px;
    background: #fafafa;
    max-height: 600px;
    overflow-y: auto;
}

.empty-state {
    color: #666;
    text-align: center;
    padding: 40px;
}

form {
    margin-bottom: 15px;
}
</style>
</head>
<body>

<h2>Select Folders & Files</h2>

<form method="post" class="controls">
    <input type="hidden" name="action" value="refresh_folders">
    <button type="submit">🔄 Refresh Tree</button>
    <?php if (!empty($config['cached_tree_updated_at'])): ?>
        <small>Last updated: <?= date('Y-m-d H:i:s', $config['cached_tree_updated_at']) ?></small>
    <?php endif; ?>
</form>

<form method="post" id="mainForm">
    <input type="hidden" name="action" value="save_folders">
    
    <div class="controls">
        <button type="button" id="expandAllBtn">Expand All</button>
        <button type="button" id="collapseAllBtn">Collapse All</button>
        <button type="button" id="selectAllBtn">Select All</button>
        <button type="button" id="deselectAllBtn">Deselect All</button>
        <button type="submit">💾 Save Selection</button>
    </div>

    <div class="tree-container">
        <?php if (empty($tree)): ?>
            <div class="empty-state">No cached data. Please refresh first.</div>
        <?php else: ?>
            <?php renderTree($tree, $selected); ?>
        <?php endif; ?>
    </div>
</form>

<script>
(function() {
    'use strict';

    // Cache DOM queries
    const checkboxes = new Map();
    const toggles = new Map();
    const childrenContainers = new Map();
    
    // Initialize maps
    document.querySelectorAll('input[type="checkbox"]').forEach(box => {
        checkboxes.set(box.dataset.id, box);
    });
    
    document.querySelectorAll('.toggle').forEach(toggle => {
        toggles.set(toggle.dataset.toggleId, toggle);
    });
    
    document.querySelectorAll('.children').forEach(container => {
        childrenContainers.set(container.dataset.childrenOf, container);
    });

    // Checkbox change handler with event delegation would be better, but keeping individual listeners for clarity
    checkboxes.forEach((checkbox, id) => {
        checkbox.addEventListener('change', function(e) {
            e.stopPropagation();
            handleCheckboxChange(id, this.checked);
        });
    });

    function handleCheckboxChange(id, isChecked) {
        // Update all descendants
        updateDescendants(id, isChecked);
        
        // Update all ancestors
        updateAncestors(id);
    }

    function updateDescendants(parentId, isChecked) {
        const descendants = getDirectChildren(parentId);
        
        descendants.forEach(childId => {
            const checkbox = checkboxes.get(childId);
            if (checkbox) {
                checkbox.checked = isChecked;
                checkbox.indeterminate = false;
                // Recursively update
                updateDescendants(childId, isChecked);
            }
        });
    }

    function updateAncestors(childId) {
        const checkbox = checkboxes.get(childId);
        if (!checkbox) return;
        
        const parentId = checkbox.dataset.parent;
        if (!parentId) return;

        const parentCheckbox = checkboxes.get(parentId);
        if (!parentCheckbox) return;

        const siblings = getDirectChildren(parentId);
        
        let allChecked = true;
        let noneChecked = true;
        
        siblings.forEach(siblingId => {
            const siblingBox = checkboxes.get(siblingId);
            if (siblingBox) {
                if (siblingBox.checked || siblingBox.indeterminate) {
                    noneChecked = false;
                }
                if (!siblingBox.checked) {
                    allChecked = false;
                }
            }
        });

        if (allChecked) {
            parentCheckbox.checked = true;
            parentCheckbox.indeterminate = false;
        } else if (noneChecked) {
            parentCheckbox.checked = false;
            parentCheckbox.indeterminate = false;
        } else {
            parentCheckbox.checked = false;
            parentCheckbox.indeterminate = true;
        }

        // Recursively update parent's ancestors
        updateAncestors(parentId);
    }

    function getDirectChildren(parentId) {
        const children = [];
        checkboxes.forEach((checkbox, id) => {
            if (checkbox.dataset.parent === parentId) {
                children.push(id);
            }
        });
        return children;
    }

    // Toggle functionality - completely separate from checkbox logic
    toggles.forEach((toggle, id) => {
        toggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            toggleNode(id);
        });
    });

    function toggleNode(id) {
        const container = childrenContainers.get(id);
        const toggle = toggles.get(id);
        
        if (!container || !toggle) return;

        const isCurrentlyHidden = container.style.display === 'none';
        
        if (isCurrentlyHidden) {
            container.style.display = 'block';
            toggle.textContent = '▸';
            toggle.classList.add('expanded');
        } else {
            container.style.display = 'none';
            toggle.textContent = '▸';
            toggle.classList.remove('expanded');
        }
    }

    // Expand all
    document.getElementById('expandAllBtn').addEventListener('click', function(e) {
        e.preventDefault();
        childrenContainers.forEach((container, id) => {
            container.style.display = 'block';
            const toggle = toggles.get(id);
            if (toggle) {
                toggle.textContent = '▸';
                toggle.classList.add('expanded');
            }
        });
    });

    // Collapse all
    document.getElementById('collapseAllBtn').addEventListener('click', function(e) {
        e.preventDefault();
        childrenContainers.forEach((container, id) => {
            container.style.display = 'none';
            const toggle = toggles.get(id);
            if (toggle) {
                toggle.textContent = '▸';
                toggle.classList.remove('expanded');
            }
        });
    });

    // Select all
    document.getElementById('selectAllBtn').addEventListener('click', function(e) {
        e.preventDefault();
        checkboxes.forEach(checkbox => {
            checkbox.checked = true;
            checkbox.indeterminate = false;
        });
    });

    // Deselect all
    document.getElementById('deselectAllBtn').addEventListener('click', function(e) {
        e.preventDefault();
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
            checkbox.indeterminate = false;
        });
    });

})();
</script>

</body>
</html>