<?php
/* ================= CONFIG ================= */

$CONFIG_FILE = __DIR__ . '/onedrive_config.json';
$STATE_FILE  = __DIR__ . '/sync_state.json';

$VERBOSE = true; // ← turn off in production

function logMsg($msg) {
    global $VERBOSE;
    if ($VERBOSE) {
        echo '[' . date('H:i:s') . '] ' . $msg . PHP_EOL;
    }
}

if (!file_exists($CONFIG_FILE)) {
    die("Config missing\n");
}

$config = json_decode(file_get_contents($CONFIG_FILE), true);
$state  = file_exists($STATE_FILE)
    ? json_decode(file_get_contents($STATE_FILE), true)
    : ['files' => [], 'last_run' => 0];

$now = time();

/* ================= LOCK ================= */

$lockFile = $config['execution']['lock_file'] ?? (__DIR__ . '/sync.lock');
$lockFp = fopen($lockFile, 'c');
if (!$lockFp || !flock($lockFp, LOCK_EX | LOCK_NB)) {
    die("Another sync is running\n");
}

/* ================= RUN INTERVAL ================= */

$minRun = $config['sync_policy']['min_run_interval_seconds'] ?? 0;
if (($now - ($state['last_run'] ?? 0)) < $minRun) {
    logMsg("Run throttled (min interval not met)");
    exit;
}

/* ================= TOKEN ================= */

function refreshToken(&$config, $CONFIG_FILE) {
    logMsg("Refreshing access token…");

    if (empty($config['refresh_token'])) {
        logMsg("No refresh token available");
        return false;
    }

    $res = @file_get_contents(
        "https://login.microsoftonline.com/{$config['tenant_id']}/oauth2/v2.0/token",
        false,
        stream_context_create([
            'http' => [
                'method'  => 'POST',
                'header'  => 'Content-Type: application/x-www-form-urlencoded',
                'content' => http_build_query([
                    'client_id'     => $config['client_id'],
                    'client_secret' => $config['client_secret'],
                    'refresh_token' => $config['refresh_token'],
                    'grant_type'    => 'refresh_token'
                ])
            ]
        ])
    );

    $t = json_decode($res, true);
    if (!empty($t['access_token'])) {
        $config['access_token']  = $t['access_token'];
        $config['refresh_token'] = $t['refresh_token'] ?? $config['refresh_token'];
        $config['token_expires'] = time() + ($t['expires_in'] ?? 3600);
        file_put_contents($CONFIG_FILE, json_encode($config, JSON_PRETTY_PRINT));
        logMsg("Token refreshed");
        return true;
    }

    logMsg("Token refresh failed");
    return false;
}


function api($endpoint, &$config, $CONFIG_FILE) {
    if (($config['token_expires'] ?? 0) < time()) {
        refreshToken($config, $CONFIG_FILE);
    }

    if (empty($config['access_token'])) {
        logMsg("No access token available for API call");
        return [];
    }

    logMsg("API GET $endpoint");

    $ctx = stream_context_create([
        'http' => [
            'header' => "Authorization: Bearer {$config['access_token']}"
        ]
    ]);

    $res = @file_get_contents("https://graph.microsoft.com/v1.0$endpoint", false, $ctx);
    if ($res === false) {
        logMsg("API call failed: $endpoint");
        return [];
    }

    return json_decode($res, true);
}


/* ================= HELPERS ================= */

function ensureDir($dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
        logMsg("Created directory: $dir");
    }
}

function shouldSyncFile($id, $etag, $state, $config) {
    if (!isset($state['files'][$id])) return true;

    $prev = $state['files'][$id];
    $min  = $config['sync_policy']['min_file_interval_seconds'] ?? 0;

    if (($prev['etag'] ?? '') !== $etag) {
        return true;
    }

    if ((time() - ($prev['synced_at'] ?? 0)) < $min) {
        logMsg("Skipping $id (cooldown)");
        return false;
    }

    return true;
}

/* ================= RECURSIVE SYNC ================= */

function syncFolderRecursive(
    $itemId,
    $remotePath,
    &$config,
    &$state,
    $CONFIG_FILE,
    $basePath,
    $tempExt,
    $maxFiles,
    &$total
) {
    logMsg("Scanning folder: $remotePath");

    $children = api("/me/drive/items/$itemId/children", $config, $CONFIG_FILE);

    foreach ($children['value'] ?? [] as $item) {
        if ($total >= $maxFiles) return;

        $id   = (string)$item['id'];
        $name = $item['name'];
        $etag = $item['eTag'] ?? '';
        $path = trim($remotePath . '/' . $name, '/');

        if (!empty($item['folder'])) {
            syncFolderRecursive(
                $id, $path,
                $config, $state, $CONFIG_FILE,
                $basePath, $tempExt, $maxFiles, $total
            );
            continue;
        }

        if (!isset($item['@microsoft.graph.downloadUrl'])) {
            logMsg("No download URL for $path");
            continue;
        }

        if (!shouldSyncFile($id, $etag, $state, $config)) continue;

        $final = $basePath . '/' . $path;
        ensureDir(dirname($final));

        $tmp = $final . $tempExt;
        $data = @file_get_contents($item['@microsoft.graph.downloadUrl']);
        if ($data === false) {
            logMsg("Download failed: $path");
            continue;
        }

        file_put_contents($tmp, $data);
        rename($tmp, $final);

        logMsg("Downloaded: $path");

        $state['files'][$id] = [
            'etag'      => $etag,
            'synced_at' => time()
        ];
        $total++;
    }
}

/* ================= RUN ================= */

$basePath = rtrim($config['local_storage']['base_path'], '/');
$tempExt  = $config['local_storage']['temp_extension'] ?? '.partial';
$maxFiles = $config['sync_policy']['max_files_per_run'] ?? PHP_INT_MAX;

$total = 0;

logMsg("Starting sync…");
logMsg("Selected folders: " . count($config['selected_folders'] ?? []));

foreach ($config['selected_folders'] ?? [] as $itemId => $path) {
    logMsg("Root folder: $path ($itemId)");

    $item = api("/me/drive/items/$itemId", $config, $CONFIG_FILE);
    if (empty($item['folder'])) {
        logMsg("Not a folder: $path");
        continue;
    }

    syncFolderRecursive(
        $itemId,
        $path,
        $config,
        $state,
        $CONFIG_FILE,
        $basePath,
        $tempExt,
        $maxFiles,
        $total
    );
}

/* ================= FINAL ================= */

$state['last_run'] = time();
file_put_contents($STATE_FILE, json_encode($state, JSON_PRETTY_PRINT));

flock($lockFp, LOCK_UN);
fclose($lockFp);

logMsg("DONE — synced $total file(s)");
